<?php
session_start();
include 'conexao.php';

// Verifica se o usuário está logado e tem o perfil correto
$perfil = strtoupper($_SESSION['usuario']['perfil'] ?? '');
if (!in_array($perfil, ['SUBDIRECÇÃO PEDAGÓGICA', 'SECRETARIA PEDAGÓGICA'])) {
    header("Location: login.php");
    exit;
}

// Configuração de data e fuso
setlocale(LC_TIME, 'pt_PT.UTF-8', 'pt_BR.UTF-8', 'portuguese');
date_default_timezone_set('Africa/Luanda');

// BUSCA DE TURMAS
$turmas = [];
$res = $conn->query("SELECT id, nome, classe, curso, turno, ano FROM turmas ORDER BY classe, nome");
if ($res) {
    while ($row = $res->fetch_assoc()) {
        $turmas[] = $row;
    }
}

// Recebe parâmetros (GET) de forma segura
$turma_id = filter_input(INPUT_GET, 'turma_id', FILTER_VALIDATE_INT);
$trimestre = filter_input(INPUT_GET, 'trimestre', FILTER_VALIDATE_INT);

$alunos = $disciplinas = $estatisticas_disciplinas = [];
$nome_turma = $classe_turma = $curso_turma = $trimestre_extenso = '';
$ano_lectivo = date('Y');

if ($turma_id) {
    $stmt_turma = $conn->prepare("SELECT nome, classe, curso, ano FROM turmas WHERE id = ?");
    $stmt_turma->bind_param("i", $turma_id);
    $stmt_turma->execute();
    $res_turma = $stmt_turma->get_result();
    if ($row_turma = $res_turma->fetch_assoc()) {
        $nome_turma = htmlspecialchars($row_turma['nome']);
        $classe_turma = htmlspecialchars($row_turma['classe']);
        $curso_turma = htmlspecialchars($row_turma['curso']);
        $ano_lectivo = $row_turma['ano'] ?? date('Y'); 
    }
}

if ($trimestre == 1) {
    $trimestre_extenso = 'I Trimestre';
} elseif ($trimestre == 2) {
    $trimestre_extenso = 'II Trimestre';
} elseif ($trimestre == 3) {
    $trimestre_extenso = 'III Trimestre';
}


// ====================================================================
// NOVO BLOCO: Lógica para Salvar e Buscar Hora/Sala
// ====================================================================

$hora_mapa = '____';
$sala_mapa = '____';

if ($turma_id && $trimestre) {
    // 1. Lógica para Salvar (Insert/Update)
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'save_mapa_info') {
        // Sanitiza os inputs
        $input_hora = filter_input(INPUT_POST, 'hora_mapa', FILTER_SANITIZE_STRING);
        $input_sala = filter_input(INPUT_POST, 'sala_mapa', FILTER_SANITIZE_STRING);

        $stmt_save = $conn->prepare("
            INSERT INTO mapa_info (turma_id, trimestre, ano_lectivo, hora, sala)
            VALUES (?, ?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE hora = VALUES(hora), sala = VALUES(sala)
        ");
        // Nota: O ano_lectivo precisa ser o valor da turma, garantindo a unicidade
        $stmt_save->bind_param("iisss", $turma_id, $trimestre, $ano_lectivo, $input_hora, $input_sala);
        if ($stmt_save->execute()) {
            // Sucesso: Atribui os novos valores
            $hora_mapa = htmlspecialchars($input_hora);
            $sala_mapa = htmlspecialchars($input_sala);
        }
        $stmt_save->close();
    }
    
    // 2. Lógica para Buscar (Fetch)
    $stmt_fetch_mapa = $conn->prepare("SELECT hora, sala FROM mapa_info WHERE turma_id = ? AND trimestre = ? AND ano_lectivo = ?");
    $stmt_fetch_mapa->bind_param("iis", $turma_id, $trimestre, $ano_lectivo);
    $stmt_fetch_mapa->execute();
    $res_fetch_mapa = $stmt_fetch_mapa->get_result();
    if ($row_mapa = $res_fetch_mapa->fetch_assoc()) {
        $hora_mapa = htmlspecialchars($row_mapa['hora']);
        $sala_mapa = htmlspecialchars($row_mapa['sala']);
    }
    $stmt_fetch_mapa->close();
}
// ====================================================================
// FIM NOVO BLOCO
// ====================================================================


// ====================================================================
// FUNÇÃO DE ORDENAÇÃO DE DISCIPLINAS (existente)
// ====================================================================
function ordenar_disciplinas($disciplinas) {
    $grupos_ordem = [
        'Linguas' => ['L.PORT', 'LINGUA PORTUGUESA', 'PORTUGUÊS', 'INGLÊS', 'FRANCÊS'],
        'Exatas' => ['MATEMÁTICA', 'MAT', 'FÍSICA', 'QUÍMICA'],
        'Biologia' => ['BIOLOGIA'],
        'Sociais' => ['HISTÓRIA', 'GEOGRAFIA', 'FILOSOFIA', 'PSICOLOGIA', 'D.I.A']
    ];
    $disciplinas_por_grupo = ['Linguas' => [], 'Exatas' => [], 'Biologia' => [], 'Sociais' => [], 'Resto' => []];
    foreach ($disciplinas as $d) {
        $nome = strtoupper(trim($d['nome']));
        $encontrado = false;
        foreach ($grupos_ordem as $grupo => $nomes_grupo) {
            if (in_array($nome, $nomes_grupo)) {
                $disciplinas_por_grupo[$grupo][] = $d;
                $encontrado = true;
                break;
            }
        }
        if (!$encontrado) {
            $disciplinas_por_grupo['Resto'][] = $d;
        }
    }
    $ordenadas = [];
    $ordem_final = ['Linguas', 'Exatas', 'Biologia', 'Sociais', 'Resto'];
    foreach ($ordem_final as $grupo) {
        usort($disciplinas_por_grupo[$grupo], function($a, $b) {
            return strcmp($a['nome'], $b['nome']);
        });
        $ordenadas = array_merge($ordenadas, $disciplinas_por_grupo[$grupo]);
    }
    return $ordenadas;
}
// ====================================================================
// INÍCIO DO CÁLCULO DAS ESTATÍSTICAS (existente)
// ====================================================================

if ($turma_id && $trimestre) {
    // 1. Buscar alunos da turma
    $stmt_alunos = $conn->prepare("SELECT id, nome, genero, abandono, transferido FROM alunos WHERE turma_id = ? ORDER BY nome");
    $stmt_alunos->bind_param("i", $turma_id);
    $stmt_alunos->execute();
    $alunos = $stmt_alunos->get_result()->fetch_all(MYSQLI_ASSOC);

    // Variáveis para contagem de Abandono e Transferidos
    $totais_movimento = [
        'abandono' => ['M' => 0, 'F' => 0, 'Total' => 0],
        'transferido' => ['M' => 0, 'F' => 0, 'Total' => 0]
    ];

    // Loop ÚNICO para calcular Abandono/Transferência
    foreach ($alunos as $aluno_mov) {
        $genero_mov = strtoupper(substr($aluno_mov['genero'] ?? 'M', 0, 1));
        $genero_mov = ($genero_mov === 'F') ? 'F' : 'M';

        if (($aluno_mov['abandono'] ?? 0) == 1) {
            $totais_movimento['abandono'][$genero_mov]++;
            $totais_movimento['abandono']['Total']++;
        }

        if (($aluno_mov['transferido'] ?? 0) == 1) {
            $totais_movimento['transferido'][$genero_mov]++;
            $totais_movimento['transferido']['Total']++;
        }
    }

    // Buscar e ordenar disciplinas
    $stmt_disciplinas = $conn->prepare("SELECT id, nome FROM disciplinas WHERE turma_id = ? ORDER BY nome");
    $stmt_disciplinas->bind_param("i", $turma_id);
    $stmt_disciplinas->execute();
    $disciplinas_brutas = $stmt_disciplinas->get_result()->fetch_all(MYSQLI_ASSOC);
    $disciplinas = ordenar_disciplinas($disciplinas_brutas);
    
    // Inicialização da estrutura de estatísticas
    $estatisticas_disciplinas = [];

    // Prepara statements (para reutilização e eficiência)
    $stmt_mac = $conn->prepare("SELECT mac FROM avaliacoes_continuas WHERE aluno_id = ? AND disciplina_id = ? AND trimestre = ?");
    $stmt_notas = $conn->prepare("SELECT npp, npt FROM notas WHERE aluno_id = ? AND disciplina_id = ? AND trimestre = ?");

    // Loop principal: Disciplinas -> Alunos
    foreach ($disciplinas as $disc) {
        $disc_id = $disc['id'];
        
        // Inicializa contadores por disciplina
        $estatisticas_disciplinas[$disc_id] = [
            'nome' => $disc['nome'],
            'matriculados' => ['M' => 0, 'F' => 0, 'Total' => 0],
            'avaliados' => ['M' => 0, 'F' => 0, 'Total' => 0],
            'aprovados' => ['M' => 0, 'F' => 0, 'Total' => 0],
            'reprovados' => ['M' => 0, 'F' => 0, 'Total' => 0],
            'aproveitamento_perc' => ['M' => 0, 'F' => 0, 'Total' => 0],
            'rendimento_perc' => ['M' => 0, 'F' => 0, 'Total' => 0],
        ];

        // Loop pelos alunos
        foreach ($alunos as $aluno) {
            $genero_col = $aluno['genero'] ?? 'M'; 
            $genero = strtoupper(substr($genero_col, 0, 1));
            $genero = ($genero === 'F') ? 'F' : 'M';

            // Contabiliza matriculados
            $estatisticas_disciplinas[$disc_id]['matriculados'][$genero]++;
            $estatisticas_disciplinas[$disc_id]['matriculados']['Total']++;

            $mac = 0; $npp = 0; $npt = 0; $media_trimestral = 0;

            // Busca MAC
            $stmt_mac->bind_param("iii", $aluno['id'], $disc_id, $trimestre);
            $stmt_mac->execute();
            $res_mac = $stmt_mac->get_result()->fetch_assoc();
            $mac = floatval($res_mac['mac'] ?? 0);

            // Busca NPP e NPT
            $stmt_notas->bind_param("iii", $aluno['id'], $disc_id, $trimestre);
            $stmt_notas->execute();
            $res_notas = $stmt_notas->get_result()->fetch_assoc();
            $npp = floatval($res_notas['npp'] ?? 0);
            $npt = floatval($res_notas['npt'] ?? 0);

            // Cálculo da Média Trimestral (MT = (MAC + NPP + NPT) / 3)
            if ($mac > 0 || $npp > 0 || $npt > 0) {
                $media_trimestral = ($mac + $npp + $npt) / 3;
                
                // Contabiliza Avaliados
                $estatisticas_disciplinas[$disc_id]['avaliados'][$genero]++;
                $estatisticas_disciplinas[$disc_id]['avaliados']['Total']++;

                // Aprovação: Média arredondada >= 10
                if (round($media_trimestral) >= 10) {
                    $estatisticas_disciplinas[$disc_id]['aprovados'][$genero]++;
                    $estatisticas_disciplinas[$disc_id]['aprovados']['Total']++;
                } else {
                    $estatisticas_disciplinas[$disc_id]['reprovados'][$genero]++;
                    $estatisticas_disciplinas[$disc_id]['reprovados']['Total']++;
                }
            }
        } // Fim Loop Alunos
        
        // 2. Cálculo de Percentagens por Disciplina e Gênero
        foreach (['M', 'F', 'Total'] as $g) {
            $aprovados = $estatisticas_disciplinas[$disc_id]['aprovados'][$g];
            $avaliados = $estatisticas_disciplinas[$disc_id]['avaliados'][$g];
            $matriculados = $estatisticas_disciplinas[$disc_id]['matriculados'][$g];

            // Aproveitamento (%): Aprovados / Avaliados
            if ($avaliados > 0) {
                $estatisticas_disciplinas[$disc_id]['aproveitamento_perc'][$g] = round(($aprovados / $avaliados) * 100, 1);
            }

            // Rendimento (%): Aprovados / Matriculados
            if ($matriculados > 0) {
                $estatisticas_disciplinas[$disc_id]['rendimento_perc'][$g] = round(($aprovados / $matriculados) * 100, 1);
            }
        }
    } // Fim Loop Disciplinas
    
    // Fecha statements
    $stmt_mac->close();
    $stmt_notas->close();
}

$data_atual = date('d/m/Y');
?>

<!DOCTYPE html>
<html lang="pt">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Estatística por Turma</title>
<style>
/* ====== LAYOUT PRINCIPAL ====== */
body {
    margin: 0;
    font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    background-color: #e9ecef;
    display: flex;
    min-height: 100vh;
    overflow-x: hidden;
}

/* ====== BARRA LATERAL ====== */
.sidebar {
    width: 230px;
    background-color: #0d1b2a;
    color: white;
    display: flex;
    flex-direction: column;
    align-items: center;
    padding-top: 20px;
}
.sidebar h2 {
    font-size: 1.2em;
    margin-bottom: 20px;
    color: #00b4d8;
}
.sidebar a {
    color: white;
    text-decoration: none;
    padding: 10px 15px;
    display: block;
    width: 90%;
    text-align: left;
    border-radius: 8px;
    transition: background 0.3s;
}
.sidebar a:hover {
    background-color: #1b263b;
}

/* ====== CONTEÚDO PRINCIPAL ====== */
.main-content {
    flex: 1;
    background-color: #f8f9fa;
    display: flex;
    flex-direction: column;
}

/* ====== BARRA SUPERIOR ====== */
.topbar {
    background-color: #007bff;
    color: white;
    padding: 12px 25px;
    font-size: 1.1em;
    font-weight: bold;
    letter-spacing: 0.5px;
    box-shadow: 0 2px 5px rgba(0,0,0,0.1);
}

/* ====== CONTAINER INTERNO ====== */
.container {
    margin: 20px auto;
    width: 95%;
    max-width: 1400px;
    background-color: #ffffff;
    padding: 20px;
    border-radius: 10px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

/* ====== FORMULÁRIO SELEÇÃO ====== */
.header-form-container {
    display: flex;
    justify-content: space-between;
    flex-wrap: wrap;
    align-items: center;
    margin-bottom: 20px;
}
form {
    display: flex;
    gap: 8px;
    flex-wrap: wrap;
}
select, button {
    padding: 7px 12px;
    border-radius: 5px;
    border: 1px solid #ccc;
    font-size: 0.9em;
}
button {
    background-color: #007bff;
    color: white;
    border: none;
    cursor: pointer;
    transition: 0.3s;
}
button:hover { background-color: #0056b3; }
#print-button {
    background-color: #6c757d;
}
#print-button:hover {
    background-color: #5a6268;
}

/* ====== CABEÇALHO DA ESCOLA ====== */
.cabecalho-container {
    text-align: center;
    margin-bottom: 10px;
}

.cabecalho-container img {
    height: 80px; 
    margin-bottom: 10px;
}

.cabecalho {
    text-align: center;
    margin-bottom: 10px;
    line-height: 1.2;
}
.cabecalho h2 {
    margin: 0;
    font-weight: 600;
    font-size: 1.3em; 
}
.cabecalho h3 {
    margin: 2px 0;
    font-weight: 400;
    font-size: 1.0em; 
}
.info-trimestre-turma {
    text-align: center;
    margin-bottom: 15px;
    font-weight: bold;
    font-size: 1.0em; 
}

/* ====== BLOCO DE INFORMAÇÕES DO MAPA (NOVO) ====== */
.mapa-aproveitamento-info {
    font-size: 1.0em;
    margin-bottom: 20px;
    line-height: 1.5;
}
.mapa-aproveitamento-info p {
    margin: 5px 0;
    text-indent: 20px;
}
.mapa-aproveitamento-info b, .mapa-editable-span {
    font-weight: bold;
    display: inline-block;
}
/* Estilo para os inputs de edição */
.mapa-aproveitamento-info input[type="text"] {
    font-weight: bold;
    border: 1px solid #ccc;
    padding: 2px 4px;
    text-align: center;
    font-size: 1em; /* Garante que o texto dentro do input tenha o mesmo tamanho */
    line-height: 1.5;
}
.edit-save-button {
    margin-top: 10px;
    padding: 5px 10px;
    font-size: 0.9em;
}

/* ====== TABELA ====== */
.table-container { overflow-x: auto; width: 100%; }
table {
    width: 100%;
    border-collapse: collapse;
    font-size: 11px;
    min-width: 1500px; 
}
th, td {
    border: 1px solid #dee2e6;
    padding: 5px;
    text-align: center;
}
thead th {
    background-color: #003366;
    color: white;
}
tbody tr:nth-child(even) { background-color: #f9f9f9; }
tbody tr:hover { background-color: #f1f1f1; }
.col-total { font-weight: bold; background-color: #e9ecef; }

/* ====== ASSINATURAS ====== */
.assinaturas {
    display: flex;
    justify-content: space-between;
    margin-top: 40px;
    font-size: 1.0em; 
}
.line {
    border-bottom: 1px solid #333;
    margin-top: 50px;
}
.data-mapa {
    text-align: right;
    margin-top: 20px;
    font-size: 1.0em; 
}

/* ====== IMPRESSÃO (PAISAGEM E FONTES MAIORES) ====== */
@media print {
    /* Define a folha de impressão para Paisagem/Horizontal */
    @page {
        size: A4 landscape;
        margin: 1cm;
    }
    
    body { 
        background-color: #fff;
        font-size: 12pt; 
    }
    /* Oculta elementos de navegação e edição */
    .sidebar, .topbar, .header-form-container, .btn-back, button, 
    .edit-save-button, #input_hora, #input_sala, #mapa-info-form input[type="hidden"] { 
        display: none !important; 
    }
    .container { 
        box-shadow: none; 
        border: none; 
        padding: 0; 
        width: 100%;
        max-width: 100%;
    }
    
    /* Garante que o texto de display seja usado na impressão */
    .mapa-editable-span {
        display: inline !important; /* Mostra o texto salvo */
        font-weight: bold;
    }

    .cabecalho-container img {
        height: 60px !important;
        display: block;
        margin: 0 auto 10px auto;
    }
    
    .mapa-aproveitamento-info {
        font-size: 10pt;
        margin-bottom: 15px;
        line-height: 1.3;
    }
    
    table { 
        font-size: 9pt;
        min-width: 100%;
    }
    th, td { 
        padding: 3px 2px;
    }
    .cabecalho h2 { font-size: 12pt; }
    .cabecalho h3, .info-trimestre-turma { font-size: 10pt; }
    .assinaturas, .data-mapa { font-size: 9pt; }
}
</style>
</head>
<body>

<div class="sidebar">
    <h2>SISMAG</h2>
    <a href="painel_subpedagogico.php">🏠 Início</a>
        <a href="logout.php">🚪 Sair</a>
</div>

<div class="main-content">
    <div class="topbar">
        Estatística por Turma
    </div>

    <div class="container">

        <div class="header-form-container">
            <form method="get">
                <label>Turma:</label>
                <select name="turma_id" required>
                    <option value="">-- Escolher Turma --</option>
                    <?php foreach ($turmas as $t): ?>
                        <option value="<?= $t['id'] ?>" <?= ($t['id'] == $turma_id ? 'selected' : '') ?>>
                            <?= htmlspecialchars($t['classe'] . ' - ' . $t['nome'] . ' - ' . $t['curso']) ?>
                        </option>
                    <?php endforeach; ?>
                </select>

                <label>Trimestre:</label>
                <select name="trimestre" required>
                    <option value="1" <?= $trimestre == 1 ? 'selected' : '' ?>>I</option>
                    <option value="2" <?= $trimestre == 2 ? 'selected' : '' ?>>II</option>
                    <option value="3" <?= $trimestre == 3 ? 'selected' : '' ?>>III</option>
                </select>

                <button type="submit">Ver Estatística</button>
                <button id="print-button" type="button" onclick="window.print()">Imprimir</button>

                <?php if ($turma_id && $trimestre): ?>
                    <a href="estatistica_global.php?turma_id=<?= $turma_id ?>&trimestre=<?= $trimestre ?>" 
                       style="font-size: 0.9em; padding: 7px 10px; border-radius: 5px; text-decoration: none; 
                              background-color: #28a745; color: white;">
                        Estatística Global
                    </a>
                <?php endif; ?>
            </form>
        </div>
        
        <?php if ($turma_id && $trimestre): ?>
            
            <div class="cabecalho-container">
                <img src="insignia/insignia.png" alt="Insígnia da Escola">
            </div>

            <div class="cabecalho">
                <h2>REPÚBLICA DE ANGOLA</h2>
                <h3>COMPLEXO ESCOLAR BG0026 PADRE EMIL FRICK-CATUMBELA</h3>
                <h3>ESTATÍSTICA DE APROVEITAMENTO POR TURMA</h3>
            </div>

            <div class="info-trimestre-turma">
                <?= htmlspecialchars($trimestre_extenso) ?> - Turma: <?= htmlspecialchars($classe_turma) ?> - <?= htmlspecialchars($nome_turma) ?> - Curso: <?= htmlspecialchars($curso_turma) ?> - Ano Letivo: <?= htmlspecialchars($ano_lectivo) ?>
            </div>

            <div class="mapa-aproveitamento-info">
                <?php
                $partes_data = explode('/', $data_atual);
                $dia_mapa = $partes_data[0];
                $mes_mapa = strftime('%B', strtotime(date('Y') . '-' . $partes_data[1] . '-01'));
                $ano_mapa = $partes_data[2];
                ?>

                <form method="post" id="mapa-info-form">
                    <input type="hidden" name="action" value="save_mapa_info">
                    <input type="hidden" name="turma_id" value="<?= htmlspecialchars($turma_id) ?>">
                    <input type="hidden" name="trimestre" value="<?= htmlspecialchars($trimestre) ?>">
                    
                    <p>ESTATÍSTICA REFERENTE AO <b><?= htmlspecialchars($trimestre_extenso) ?></b> DO ANO LECTIVO <b><?= htmlspecialchars($ano_lectivo) ?></b></p>
                    
                    <p>Aos <b><?= htmlspecialchars($dia_mapa) ?></b> de <b><?= htmlspecialchars(ucfirst($mes_mapa)) ?></b> de <b><?= htmlspecialchars($ano_mapa) ?></b>, pelas 
                        <span id="display_hora" class="mapa-editable-span"><?= $hora_mapa ?></span>
                        <input type="text" id="input_hora" name="hora_mapa" value="<?= $hora_mapa ?>" style="display:none; width: 60px;">
                        horas, na sala nº 
                        <span id="display_sala" class="mapa-editable-span"><?= $sala_mapa ?></span>
                        <input type="text" id="input_sala" name="sala_mapa" value="<?= $sala_mapa ?>" style="display:none; width: 100px;">
                        do Complexo Escolar, elaborou-se o <b>Mapa de Aproveitamento</b> trimestral da <b>Turma <?= htmlspecialchars($nome_turma) ?></b>, <b><?= htmlspecialchars($classe_turma) ?></b> Classe.</p>
                    
                    <p>Curso <b><?= htmlspecialchars($curso_turma) ?></b></p>
                    <p>Obteve-se a seguinte estatística de Aproveitamento por disciplinas:</p>
                    
                    <button type="button" id="edit-mapa-button" class="edit-save-button">✏️ Editar Hora/Sala</button>
                    <button type="submit" id="save-mapa-button" class="edit-save-button" style="display:none; background-color: #28a745;">💾 Salvar</button>
                </form>
            </div>
            <?php if (empty($alunos) || empty($disciplinas)): ?>
                <div style="text-align: center; margin-top: 20px; font-size: 1.1em; color: #cc3300;">
                    Nenhum aluno ou disciplina encontrada para os filtros selecionados.
                </div>
            <?php else: ?>

                <div class="table-container">
                    <table>
                        <thead>
                            <tr>
                                <th rowspan="3">Nº Ord.</th>
                                <th rowspan="3">Disciplinas</th>
                                
                                <th colspan="3">Alunos Matriculados</th>
                                <th colspan="3">Alunos Avaliados</th>
                                <th colspan="3">Alunos Aprovados</th>
                                <th colspan="3">Alunos Reprovados</th>
                                <th colspan="3">Aproveitamento (%)</th>
                                <th colspan="3">Rendimento (%)</th>
                                <th colspan="3">Abandono</th> 
                                <th colspan="3">Transferidos</th> 
                            </tr>
                            <tr>
                                <?php foreach (range(1, 8) as $i): ?>
                                    <th colspan="3"></th> 
                                <?php endforeach; ?>
                            </tr>
                            <tr>
                                <?php foreach (range(1, 8) as $i): ?>
                                    <th>M</th>
                                    <th>F</th>
                                    <th class="col-total">Total</th>
                                <?php endforeach; ?>
                            </tr>
                        </thead>
                        <tbody>
                            <?php 
                            $n_ord = 1; 
                            $grand_totals = [
                                'matriculados' => ['M' => 0, 'F' => 0, 'Total' => 0],
                                'aprovados' => ['M' => 0, 'F' => 0, 'Total' => 0],
                                'avaliados' => ['M' => 0, 'F' => 0, 'Total' => 0],
                            ];
                            
                            foreach ($estatisticas_disciplinas as $stats): 
                                // Soma para os Totais Gerais
                                $grand_totals['matriculados']['M'] += $stats['matriculados']['M'];
                                $grand_totals['matriculados']['F'] += $stats['matriculados']['F'];
                                $grand_totals['matriculados']['Total'] += $stats['matriculados']['Total'];
                                
                                $grand_totals['aprovados']['M'] += $stats['aprovados']['M'];
                                $grand_totals['aprovados']['F'] += $stats['aprovados']['F'];
                                $grand_totals['aprovados']['Total'] += $stats['aprovados']['Total'];
                                
                                $grand_totals['avaliados']['M'] += $stats['avaliados']['M'];
                                $grand_totals['avaliados']['F'] += $stats['avaliados']['F'];
                                $grand_totals['avaliados']['Total'] += $stats['avaliados']['Total'];
                            ?>
                                <tr>
                                    <td><?= $n_ord++ ?></td>
                                    <td style="text-align: left;"><?= htmlspecialchars($stats['nome']) ?></td>
                                    
                                    <td><?= $stats['matriculados']['M'] ?></td>
                                    <td><?= $stats['matriculados']['F'] ?></td>
                                    <td class="col-total"><?= $stats['matriculados']['Total'] ?></td>
                                    
                                    <td><?= $stats['avaliados']['M'] ?></td>
                                    <td><?= $stats['avaliados']['F'] ?></td>
                                    <td class="col-total"><?= $stats['avaliados']['Total'] ?></td>

                                    <td><?= $stats['aprovados']['M'] ?></td>
                                    <td><?= $stats['aprovados']['F'] ?></td>
                                    <td class="col-total"><?= $stats['aprovados']['Total'] ?></td>
                                    
                                    <td><?= $stats['reprovados']['M'] ?></td>
                                    <td><?= $stats['reprovados']['F'] ?></td>
                                    <td class="col-total"><?= $stats['reprovados']['Total'] ?></td>

                                    <td><?= $stats['aproveitamento_perc']['M'] ?></td>
                                    <td><?= $stats['aproveitamento_perc']['F'] ?></td>
                                    <td class="col-total"><?= $stats['aproveitamento_perc']['Total'] ?></td>
                                    
                                    <td><?= $stats['rendimento_perc']['M'] ?></td>
                                    <td><?= $stats['rendimento_perc']['F'] ?></td>
                                    <td class="col-total"><?= $stats['rendimento_perc']['Total'] ?></td>
                                    
                                    <td></td>
                                    <td></td>
                                    <td class="col-total"></td>
                                    
                                    <td></td>
                                    <td></td>
                                    <td class="col-total"></td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <div class="data-mapa">
                    Catumbela, aos <?= $data_atual ?>
                </div>

                <div class="assinaturas">
                    <div class="assinatura-esquerda">
                        <p>Assinatura do Director de Turma</p>
                        <div class="line"></div>
                    </div>
                    <div class="assinatura-direita">
                        <p>Assinatura do Subdirecto Pedagógico</p>
                        <div class="line"></div>
                    </div>
                </div>

            <?php endif; ?>
        <?php endif; ?>
        
        </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    var editButton = document.getElementById('edit-mapa-button');
    var saveButton = document.getElementById('save-mapa-button');
    
    // Elementos para Hora
    var displayHora = document.getElementById('display_hora');
    var inputHora = document.getElementById('input_hora');
    
    // Elementos para Sala
    var displaySala = document.getElementById('display_sala');
    var inputSala = document.getElementById('input_sala');

    // Só inicia o controle se a turma e trimestre estiverem selecionados (botões existem)
    if (editButton) {
        
        // Inicializa o modo de visualização (display)
        function switchToViewMode() {
             editButton.style.display = 'inline-block';
             saveButton.style.display = 'none';

             // Mostra o texto e esconde o input
             displayHora.style.display = 'inline-block';
             inputHora.style.display = 'none';
             
             displaySala.style.display = 'inline-block';
             inputSala.style.display = 'none';

             // Atualiza o texto de exibição com o valor do input (caso tenha sido salvo ou editado)
             displayHora.innerText = inputHora.value;
             displaySala.innerText = inputSala.value;
        }
        
        // Função para ativar o modo de edição
        function switchToEditMode() {
            editButton.style.display = 'none';
            saveButton.style.display = 'inline-block';

            // Esconde o texto e mostra o input
            displayHora.style.display = 'none';
            inputHora.style.display = 'inline-block';
            
            displaySala.style.display = 'none';
            inputSala.style.display = 'inline-block';
            
            inputHora.focus();
        }
        
        editButton.addEventListener('click', switchToEditMode);

        // Se a página for carregada após um POST de sucesso, o PHP já atualizou,
        // mas para garantir, iniciamos no modo de visualização.
        switchToViewMode();
    }
});
</script>
</body>
</html>